/**
* \file: partition_fsm.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <sys/mount.h>
#include <unistd.h>

#include "mount.h"
#include "automounter.h"

#include "utils/logger.h"
#include "utils/helper.h"

#include "control/partition_fsm.h"
#include "control/device_fsm.h"
#include "control/configuration.h"

#include "model/partition_list.h"

static void partition_fsm_enter_mount_err(partition_t *partition, error_code_t err);
static void partition_fsm_enter_invalid(partition_t *partition);
static void partition_fsm_enter_mounted(partition_t *partition, bool writable, error_code_t result);

static void partition_fsm_mounting_callback_func(partition_t *partition, error_code_t return_val,
        const mount_options_t *options);
static void partition_fsm_remounting_callback_func(partition_t *partition, error_code_t return_val,
        const mount_options_t *options);
static bool partition_fsm_is_mounted_writable(const mount_options_t *options);

static void partition_fsm_unmounting_callback_func(partition_t *partition, error_code_t return_val);

static void partition_dump_fsm_err(partition_state_t coming_from, 
        partition_state_t going_to);

//--------------------------------------- API members ------------------------------------
void partition_fsm_enter_unsupported(partition_t *partition,
		partition_unsupported_reason_t unsupported_reason)
{
    partition_state_t coming_from;

    coming_from=partition_get_state(partition);

    partition_set_state_unsupported(partition,unsupported_reason);
    logger_log_debug("PARTITION_FSM - Partition %s in state unsupported. Reason: %s",partition_get_id(partition),
    		partition_get_unsupported_reason_str(partition));

    if (coming_from==PARTITION_DETECTED)
        device_fsm_signal_part_handled(partition);
    else
        partition_dump_fsm_err(coming_from,PARTITION_UNSUPPORTED);
}

void partition_fsm_enter_mounting(partition_t *partition, const char* mount_point)
{
    error_code_t result;
    mount_options_t mount_opt;
    partition_state_t coming_from;
    coming_from=partition_get_state(partition);

    if (coming_from==PARTITION_DETECTED)
    {
        if (mount_point!=NULL)
        {
        	mount_opt.mount_flags=0;
        	mount_opt.mount_options=configuration_get_automount_options(partition_get_mountfs(partition));
            partition_set_state_mounting(partition,mount_point);
            logger_log_debug("PARTITION_FSM - Partition %s in state mounting",partition_get_id(partition));

            result=mount_start_mount(partition,&mount_opt, partition_fsm_mounting_callback_func);

            if (result != RESULT_OK)
                partition_fsm_enter_mount_err(partition,result);
        }
        else
            partition_fsm_enter_mount_err(partition,RESULT_INVALID_MOUNTPOINT);
    }
    else
        partition_dump_fsm_err(coming_from,PARTITION_MOUNTING);
}

void partition_fsm_signal_part_removed(partition_t *partition)
{
    partition_fsm_enter_invalid(partition);
}

error_code_t partition_fsm_signal_unmount_request(partition_t *partition,
        partition_state_change_callback_t callback_func, void *callback_data)
{
    error_code_t result;
    partition_state_t coming_from;
    coming_from=partition_get_state(partition);

    if (coming_from==PARTITION_MOUNTED)
    {
        partition_set_state_unmounting(partition);
        logger_log_debug("PARTITION_FSM - Partition %s in state unmounting.",
                partition_get_id(partition));

        partition_set_state_change_monitor(partition,callback_func,callback_data);

        result=mount_start_unmount(partition, partition_fsm_unmounting_callback_func);
        if (result==RESULT_NORESOURCE)
            automounter_die_on_resource_issues();

        //something went wrong, back to state mounted without changing any attributes
        if (result!=RESULT_OK)
        {
            //prevent that we get the error event twice (by return value and by the callback)
            partition_remove_state_change_monitor(partition);
            partition_return_to_state_mounted_on_err(partition, result);
        }
    }
    else
        result=RESULT_NOT_MOUNTED;

    return result;
}

error_code_t partition_fsm_signal_remount_request(partition_t *partition,
        const mount_options_t *options,partition_state_change_callback_t callback_func, void *callback_data)
{
    error_code_t result;
    mount_options_t mount_opt;
    partition_state_t coming_from;

    coming_from=partition_get_state(partition);
    if (coming_from==PARTITION_MOUNTED)
    {
        mount_opt.mount_options=options->mount_options;
        mount_opt.mount_flags=options->mount_flags;
        mount_opt.mount_flags |= MS_REMOUNT;

        partition_set_state_remounting(partition);
        logger_log_debug("PARTITION_FSM - Partition %s in state remounting.",
                partition_get_id(partition));

        partition_set_state_change_monitor(partition,callback_func,callback_data);
        result=mount_start_mount(partition,&mount_opt, partition_fsm_remounting_callback_func);
        if (result==RESULT_NORESOURCE)
        	automounter_die_on_resource_issues();

        //something went wrong, back to state mounted without changing any attributes
        if (result != RESULT_OK)
        {
            //prevent that we get the error event twice (by return value and by the callback)
            partition_remove_state_change_monitor(partition);
            partition_return_to_state_mounted_on_err(partition,result);
        }
    }
    else
    {
        partition_dump_fsm_err(coming_from,PARTITION_REMOUNTING);
        result=RESULT_NOT_MOUNTED;
    }
    return result;
}
//----------------------------------------------------------------------------------------


//--------------------------------------- internal members -------------------------------
static void partition_fsm_enter_mount_err(partition_t *partition, error_code_t err)
{
    partition_state_t coming_from;
    const char *mount_point;

    coming_from=partition_get_state(partition);

    if (coming_from==PARTITION_MOUNTING || coming_from==PARTITION_DETECTED)
    {
    	mount_point=partition_get_mountpoint(partition);
    	if (mount_point!=NULL)
    		(void)rmdir(mount_point);
        partition_set_state_mount_err(partition,err);
        logger_log_debug("PARTITION_FSM - Partition %s in state mount error (error code=%d)",
                partition_get_id(partition), partition_get_error(partition));
        device_fsm_signal_part_handled(partition);
    }
    else
        partition_dump_fsm_err(coming_from, PARTITION_MOUNT_ERR);
}

static void partition_fsm_mounting_callback_func(partition_t *partition, error_code_t return_val,
        const mount_options_t *options)
{
    bool writable;

    if (return_val==RESULT_OK)
    {
        //guard: no_error
        writable=partition_fsm_is_mounted_writable(options);
        partition_fsm_enter_mounted(partition,writable,RESULT_OK);
    }
    else
        //guard: mount_error
        partition_fsm_enter_mount_err(partition,return_val);
}

static void partition_fsm_remounting_callback_func(partition_t *partition, error_code_t return_val,
        const mount_options_t *options)
{
    bool writable;
    writable=partition_fsm_is_mounted_writable(options);
    partition_fsm_enter_mounted(partition,writable,return_val);
}

static bool partition_fsm_is_mounted_writable(const mount_options_t *options)
{
    if ((options->mount_flags & MS_RDONLY) != 0)
        return false;

    if (options->mount_options==NULL)
        return false;

    return helper_does_string_contain_token(options->mount_options, "rw",", \t");
}

static void partition_fsm_unmounting_callback_func(partition_t *partition, error_code_t return_val)
{
    if (return_val==RESULT_OK)
    {
        partition_set_state_unmounted(partition);
        logger_log_debug("PARTITION_FSM - Partition %s in state unmounted",partition_get_id(partition));
    }
    else if (return_val==RESULT_NORESOURCE)
        automounter_die_on_resource_issues();
    else
        partition_return_to_state_mounted_on_err(partition,return_val);

    device_fsm_signal_part_handled(partition);
}

static void partition_fsm_enter_mounted(partition_t *partition, bool writable, error_code_t result)
{
    partition_state_t coming_from;
    coming_from=partition_get_state(partition);

    //check for failures in the fsm
    if (coming_from==PARTITION_MOUNTING || coming_from==PARTITION_REMOUNTING)
    {
        if (result==RESULT_OK)
        {
            partition_set_state_mounted(partition,writable);
            logger_log_debug("PARTITION_FSM - Partition %s in state mounted",partition_get_id(partition));
        }
        else
        {
            partition_return_to_state_mounted_on_err(partition,result);
            logger_log_debug("PARTITION_FSM - Partition %s in state mounted(error)",partition_get_id(partition));
        }
        device_fsm_signal_part_handled(partition);
    }
    else
        partition_dump_fsm_err(coming_from,PARTITION_MOUNTED);
}

static void partition_fsm_enter_invalid(partition_t *partition)
{
    const char *mount_point;
    partition_state_t coming_from;

    coming_from=partition_get_state(partition);
    mount_point=partition_get_mountpoint(partition);

    //if we are currently working on something, signal the mount thread that we are invalid
    //from now on. This way, the mount thread will not touch anything of the model anymore so that
    //we can easily remove the partition from the model. The thread itself is cleaning up for us
    //after it is done (umounting & removing the dir).
    if (coming_from==PARTITION_MOUNTING || coming_from==PARTITION_UNMOUNTING ||
            coming_from==PARTITION_REMOUNTING)
        mount_mark_partition_invalid(partition);

    partition_set_state_invalid(partition);

    logger_log_debug("PARTITION_FSM - Partition %s in state invalid",partition_get_id(partition));

    //if we have mounted something, unmount now
    if (coming_from==PARTITION_MOUNTED)
    {
        //don't use a thread here, since the device is already gone, umount is not
        //expected to take too long.
        mount_unmount_sync(mount_point);
    }

    device_delete_partition(partition);
}

static void partition_dump_fsm_err(partition_state_t coming_from, 
        partition_state_t going_to)
{
    logger_log_error("PARTITION_FSM - We detected an error in the partition FSM."
            "Requested State change: %s -> %s",
            partition_get_state_str(coming_from),partition_get_state_str(going_to));
}
//----------------------------------------------------------------------------------------
